(in-package "CL-USER")

;this contains functions for simplifying a desc. it inlines functions
; and definitions, and reduces the init, trans, and spec formulas to
; formulas containing and, not, <->, next, and bit
; functions. applications of bit are pushed all the way down to the
; variable level. since init, trans, and spec must be of type 1, all
; subformulas will eventually be able to be converted to forms of type
; 1. however, in the meantime, formulas of type greater than 1 are
; represented by concatinating formulas of type 1.

;***cat***

(declaim (ftype (function (list) vec) cat-form))
(defun cat-form (args)
  (declare (type list args))
  (let ((n 0))
    (dolist (arg args)
      (declare (type form-vec arg))
      (if (vec-p arg)
	  (incf n (vec-num-bits arg))
	(incf n)))
    (let ((vec (new-vec n))
	  (i (1- n)))
      (dolist (arg args (make-unique-vec vec))
	(declare (type form-vec arg))
	(cond ((vec-p arg)
	       (let ((dim (vec-num-bits arg)))
		 (decf i (1- dim))
		 (dotimes (j dim (decf i)) (vec-set-bit vec (+ i j) (vec-get-bit arg j)))))
	      (t
	       (vec-set-bit vec i arg)
	       (decf i)))))))

;***mv***

(declaim (ftype (function (form-type list) formula) mv-form))
(defun mv-form (tp args)
  (make-unique-formula :fn 'mv
		       :type tp
		       :args args))

;given 0 or 1, this function returns the correct form (*zero* or
;*one*, defined in dbmc-structs)
(declaim (ftype (function (bit) formula)))
(defun sb-const-form (bit)
  (declare (type bit bit))
  (if (= bit 0) *zero* *one*))

(declaim (ftype (function (list) form-vec) const-form))
(defun const-form (bits)
  (declare (type list bits))
  (if (endp (cdr bits))
      (sb-const-form (car bits))
    (cat-form (mapcar #'sb-const-form bits))))

(declaim (ftype (function (list fixnum)) get-bit-list))
(defun get-bit-list (list bit)
  (declare (type list list) (type fixnum bit))
  (mapcar (lambda (x) 
	    (declare (type form-vec x))
	    (get-bit x bit))
	  list))

;given a simplified form, possibly with cats in it, return the form
;corresponding to the given bit. in other words, this pushes bit down
;to the variable level.
(declaim (ftype (function (form-vec fixnum) formula) get-bit))
(defun get-bit (fv bit)
  (declare (type form-vec fv) (type fixnum bit))
  (cond ((vec-p fv) (vec-get-bit fv bit))
	(t fv)))

(declaim (ftype (function (form-vec fixnum fixnum) form-vec) get-bits))
(defun get-bits (fv lb hb)
  (declare (type form-vec fv) (type fixnum lb) (type fixnum hb))
  (if (= lb hb)
      (get-bit fv lb)
    (let ((vec (new-vec (1+ (- hb lb)))))
      (dotimes (i (1+ (- hb lb)) (make-unique-vec vec))
	(vec-set-bit vec i (get-bit fv (+ lb i)))))))

;--------------------------------------------------------------------

(declaim (ftype (function (formula form-vec fixnum) form-vec) get-form))
(defun get-form (mem addr nw)
  (declare (type formula mem) (type form-vec addr) (type fixnum nw))
  (let* ((nb (* nw (the integer (third (formula-type mem)))))
	 (gf (make-unique-formula :fn 'get :type `(bv ,nb) :args (list mem addr nw))))
    (if (= nb 1)
	(make-unique-formula :fn 'bit :type '(bv 1) :args (list gf 0))
      (let ((vec (new-vec nb)))
	(dotimes (i nb (make-unique-vec vec))
	  (vec-set-bit vec i (make-unique-formula :fn 'bit :type '(bv 1) :args (list gf i))))))))

(declaim (ftype (function (formula form-vec form-vec) formula) set-form))
(defun set-form (mem addr val)
  (declare (type formula mem) (type form-vec addr) (type form-vec val))
  (make-unique-formula :fn 'set :type (formula-type mem) :args (list mem addr val)))
					      
;***ext***
(declaim (ftype (function (fixnum form-vec) vec) ext-form))
(defun ext-form (bits arg)
  (declare (type fixnum bits) (type form-vec arg))
  (let ((vec (new-vec bits))
	(vb (vec-num-bits arg)))
    (dotimes (i vb) (vec-set-bit vec i (get-bit arg i)))
    (let ((hb (get-bit arg (1- vb))))
      (loop for i from vb to (1- bits)
	    do (vec-set-bit vec i hb)))
    (make-unique-vec vec)))

;***var***
(declaim (ftype (function (formula list) form-vec) var-form))
(defun var-form (form env)
  (declare (type formula form) (type list env))
  (let ((val-pair (assoc (the symbol (car (formula-args form))) env)))
    (cond (val-pair (the form-vec (cdr val-pair)))
	  (t *zero*))))

;***and***

(declaim (ftype (function (formula formula) boolean) negation-of))
(defun negation-of (form1 form2)
  (declare (type formula form1 form2))
  (or (and (eq (formula-fn form1) 'not)
	   (eq (car (formula-args form1))
	       form2))
      (and (eq (formula-fn form2) 'not)
	   (eq (car (formula-args form2))
	       form1))))

(declaim (ftype (function (list) list) sort-and-elim-dups))
(defun sort-and-elim-dups (forms)
  (declare (type list forms))
  (let ((nforms (sort (copy-list forms) #'< :key #'formula-value))
	(nnforms nil)
	(cur nil))
    (dolist (nform nforms nnforms)
      (declare (type formula nform))
      (unless (eq nform cur)
	(setf nnforms (cons nform nnforms)))
      (setf cur nform))))

(declaim (ftype (function (list) list) and-flatten-args))
(defun and-flatten-args (args)
  (declare (type list args))
  (let ((nargs nil))
    (dolist (arg args nargs)
      (declare (type formula arg))
      (if (eq (formula-fn arg) 'and)
	  (setf nargs (append (formula-args arg) nargs))
	(setf nargs (cons arg nargs))))))

;; returns the args for the and-form.
(declaim (ftype (function (list) (or (cons formula cons) formula)) and-args))
(defun and-args (args)
  (declare (type list args))
  (if (endp args)
      *one*
    (let* ((nargs nil)
	   (args (sort (copy-list args) #'< :key #'formula-value))
	   (cur *junk*))
      (dolist (arg args (cond ((endp nargs) *one*)
			      ((endp (cdr nargs)) (the formula (car nargs)))
			      (t nargs)))
	(declare (type formula arg))
	(let ((fn (formula-fn arg)))
	  (cond ((eq fn 'const)
		 (when (eq arg *zero*) 
		   (return *zero*)))
		((negation-of cur arg)
		 (return *zero*))
		((not (eq cur arg))
		 (setf nargs (cons arg nargs))))
	  (setf cur arg))))))

(declaim (ftype (function (list) formula) sb-and-form))
(defun sb-and-form (args)
  (declare (type list args))
  (let ((nargs (and-args args)))
    (if (atom nargs)
	nargs
      (make-unique-formula :fn 'and
			   :args nargs))))

(declaim (ftype (function ((cons form-vec list)) form-vec) and-form))
(defun and-form (args)
  (declare (type (cons form-vec list) args))
  (let ((arg (car args)))
    (if (vec-p (car args))
	(let* ((bits (vec-num-bits arg))
	       (vec (new-vec bits)))
	  (dotimes (i bits (make-unique-vec vec))
	    (vec-set-bit vec i (sb-and-form (get-bit-list args i)))))
      (sb-and-form args))))

;***not***
(declaim (ftype (function (formula) formula) sb-not-form))
(defun sb-not-form (arg)
  (declare (type formula arg))
  (let* ((fn (formula-fn arg)))
    (cond ((eq fn 'const)
	   (if (eq arg *zero*) *one* *zero*))
	  ((eq fn 'not)
	   (the formula (car (formula-args arg))))
	  (t
	   (make-unique-formula :fn 'not
				:args (list arg))))))

(declaim (ftype (function (form-vec) form-vec) not-form))
(defun not-form (arg)
  (declare (type form-vec arg))
  (if (vec-p arg)
      (let* ((bits (vec-num-bits arg))
	     (vec (new-vec bits)))
	(dotimes (i bits (make-unique-vec vec))
	  (vec-set-bit vec i (sb-not-form (get-bit arg i)))))
    (sb-not-form arg)))


;***or***

(declaim (ftype (function (list) formula) sb-or-form))
(defun sb-or-form (args)
  (declare (type list args))
  (sb-not-form (sb-and-form (mapcar #'sb-not-form args))))

(declaim (ftype (function ((cons form-vec list)) form-vec) and-form))
(defun or-form (args)
  (declare (type (cons form-vec list) args))
  (if (vec-p (car args))
      (let* ((bits (vec-num-bits (car args)))
	     (vec (new-vec bits)))
	(dotimes (i bits (make-unique-vec vec))
	  (vec-set-bit vec i (sb-or-form (get-bit-list args i)))))
    (sb-or-form args)))

;***<->***

(declaim (ftype (function (formula formula) formula) sb-equiv-form1))
(defun sb-equiv-form1 (arg0 arg1)
  (declare (type formula arg0) (type formula arg1))
  (cond ((eq arg0 *one*) arg1)
	((eq arg0 *zero*) (sb-not-form arg1))
	((eq arg1 *one*) arg0)
	((eq arg1 *zero*) (sb-not-form arg0))
	((eq arg0 arg1)
	 *one*)
	((negation-of arg0 arg1)
	 *zero*)
	((eq (formula-fn arg0) 'not)
	 (if (eq (formula-fn arg1) 'not)
	     (make-unique-formula :fn '<->
				  :args (sortforms (list (sb-not-form arg0)
							 (sb-not-form arg1))))
	   (sb-not-form (make-unique-formula :fn '<->
					     :args (sortforms (list (sb-not-form arg0)
								    arg1))))))
	((eq (formula-fn arg1) 'not)
	 (sb-not-form (make-unique-formula :fn '<->
					   :args (sortforms (list arg0 
								  (sb-not-form arg1))))))
	(t
	 (make-unique-formula :fn '<->
			      :args (sortforms (list arg0 arg1))))))

(declaim (ftype (function (formula list) list) equiv-leaves1))
(defun equiv-leaves1 (form leaves)
  (declare (type formula form) (type list leaves))
  (case (formula-fn form)
    (<-> (equiv-leaves1 (second (formula-args form))
			(equiv-leaves1 (first (formula-args form)) leaves)))
    (not (cons (car (formula-args form)) leaves))
    (otherwise (cons form leaves))))

(declaim (ftype (function (formula) list) equiv-leaves))
(defun equiv-leaves (form)
  (declare (type formula form))
  (case (formula-fn form)
    (<-> (equiv-leaves1 (second (formula-args form))
			(equiv-leaves1 (first (formula-args form)) nil)))
    (not (if (eq (formula-fn (car (formula-args form)))
		 '<->)
	     (equiv-leaves1 (car (formula-args form)) nil)
	   (formula-args form)))
    (otherwise (list form))))

(declaim (ftype (function (formula list)) remove-doubles))
(defun remove-doubles1 (form doubles)
  (case (formula-fn form)
    (<-> (sb-equiv-form1 (remove-doubles1 (first (formula-args form)) doubles)
			 (remove-doubles1 (second (formula-args form)) doubles)))
    (not (if (member (first (formula-args form)) doubles :test 'eq)
	     *zero*
	   form))
    (otherwise (if (member form doubles :test 'eq)
		   *one*
		 form))))

(declaim (ftype (function (formula list)) remove-doubles))
(defun remove-doubles (form doubles)
  (declare (type formula form) (type list doubles))
  (case (formula-fn form)
    (<-> (sb-equiv-form1 (remove-doubles1 (first (formula-args form)) doubles)
			 (remove-doubles1 (second (formula-args form)) doubles)))
    (not (cond ((eq (formula-fn (car (formula-args form))) '<->)
		(not-form (remove-doubles1 (car (formula-args form)) doubles)))
	       ((member (car (formula-args form)) doubles :test 'eq)
		*zero*)
	       (t form)))
    (otherwise (if (member form doubles :test 'eq)
		   *one*
		 form))))

(declaim (ftype (function ((cons formula (cons formula null))) formula) sb-equiv-form))
(defun sb-equiv-form (args)
  (declare (type (cons formula (cons formula null)) args))
  (let* ((arg1 (first args))
	 (arg2 (second args))
	 (doubles (intersection (equiv-leaves arg1) (equiv-leaves arg2) :test 'eq)))
    (sb-equiv-form1 (remove-doubles arg1 doubles)
		    (remove-doubles arg2 doubles))))

(declaim (ftype (function ((cons form-vec (cons formula null))) form-vec)
		equiv-form))
(defun equiv-form (args)
  (declare (type (cons form-vec (cons form-vec null))))
  (if (vec-p (car args))
      (let* ((bits (vec-num-bits (car args)))
	     (vec (new-vec bits)))
	(dotimes (i bits (make-unique-vec vec))
	  (vec-set-bit vec i (sb-equiv-form (get-bit-list args i)))))
    (sb-equiv-form args)))

;***=***

(declaim (ftype (function (list list list) formula) mv-equals-form))
(defun mv-equals-form (args1 args2 nargs)
  (declare (type list args1) (type list args2) (type list nargs))
  (if (endp args1)
      (sb-and-form nargs)
    (mv-equals-form (cdr args1) (cdr args2)
		    (cons (equals-form (list (car args1) (car args2)))
			   nargs))))

(declaim (ftype (function ((cons form-vec (cons form-vec null))) formula) equals-form))
(defun equals-form (args)
  (let ((arg1 (first args))
	(arg2 (second args)))
    (cond ((eq arg1 arg2) *one*)
	  ((vec-p arg1)	;;bit vector
	   (let ((nargs nil))
	     (dotimes (i (vec-num-bits arg1) (sb-and-form nargs))
	       (setf nargs (cons (sb-equiv-form (get-bit-list args i)) nargs)))))
	  ((eq (first (formula-type arg1)) 'mv) ;; mv formula
	   (mv-equals-form (formula-args arg1) (formula-args arg2) nil))
	  ((eq (first (formula-type arg1)) 'mem)
	   (make-unique-formula :fn '= :args args))
	  (t ;; single bits
	   (sb-equiv-form args)))))

;***xor***

(declaim (ftype (function (list list) list) sb-xor-form1))
(defun sb-xor-form1 (args nargs)
  (cond ((endp args) nargs)
	((endp (cdr args)) (cons (the formula (car args)) nargs))
	(t (let ((arg1 (first args))
		 (arg2 (second args)))
	     (declare (type formula arg1) (type formula arg2))
	     (sb-xor-form1 (cddr args) 
			   (cons (sb-not-form (sb-equiv-form (list arg1 arg2)))
				 nargs))))))

(declaim (ftype (function (list) formula) sb-xor-form))
(defun sb-xor-form (args)
  (declare (type list args))
  (if (endp args)
      *zero*
    (let ((nargs args))
      (loop until (endp (cdr nargs))
	    do (setf nargs (sb-xor-form1 nargs nil)))
      (car nargs))))

(declaim (ftype (function ((cons form-vec list)) form-vec)
		xor-form))
(defun xor-form (args)
  (declare (type (cons form-vec list) args))
  (if (vec-p (car args))
      (let* ((nb (vec-num-bits (car args)))
	     (vec (new-vec nb)))
	(dotimes (i nb (make-unique-vec vec))
	  (vec-set-bit vec i (sb-xor-form (get-bit-list args i)))))
    (sb-xor-form args)))

;***+/-/inc/dec***

(declaim (ftype (function (formula formula formula) (values formula formula)) fa))
(defun fa (a b c)
  (declare (type formula a) (type formula b) (type formula c))
  (values (sb-xor-form (list a b c))
	  (sb-or-form (list (sb-and-form (list a b))
			    (sb-and-form (list a c))
			    (sb-and-form (list b c))))))

(declaim (ftype (function (formula formula) (values formula formula)) ha))
(defun ha (a b)
  (declare (type formula a) (type formula b))
  (values (sb-xor-form (list a b))
	  (sb-and-form (list a b))))

(declaim (ftype (function (heap heap) formula)))
(defun add-slice (cheap nheap)
  (declare (type heap cheap) (type heap nheap))
  (case (heap-count cheap)
    (0 *zero*)
    (1 (heap-remove cheap))
    (2 (multiple-value-bind
	 (s c)
	 (ha (heap-remove cheap) (heap-remove cheap))
	 (heap-insert nheap c)
	 s))
    (3 (multiple-value-bind
	 (s c)
	 (fa (heap-remove cheap) (heap-remove cheap) (heap-remove cheap))
	 (heap-insert nheap c)
	 s))
    (otherwise
     (multiple-value-bind
	 (s c)
	 (fa (heap-remove cheap) (heap-remove cheap) (heap-remove cheap))
	 (heap-insert nheap c)
	 (heap-insert cheap s)
	 (add-slice cheap nheap)))))

(declaim (ftype (function (heap list) list) heap-insert-list))
(defun heap-insert-list (heap lst)
  (declare (type heap heap) (type list lst))
  (dolist (x lst lst) (heap-insert heap x)))

;;   (when (consp lst)
;;     (heap-insert heap (car lst))
;;     (heap-insert-list heap (cdr lst))
;;     lst))

;; (declaim (ftype (function (list fixnum heap heap list) list) add-slices-loop))
;; (defun add-slices-loop (slices len cheap nheap nargs)
;;   (declare (type list slices) (type fixnum len) (type heap cheap)
;; 	   (type heap nheap) (type list nargs))
;;   (if (<= len 0)
;;       nargs
;;     (progn
;;       (when (consp slices)
;; 	(heap-insert-list nheap (car slices)))
;;       (add-slices-loop (cdr slices) (1- len) nheap cheap
;; 		 (cons (if (heap-empty-p cheap)
;; 			   *zero*
;; 			 (add-slice cheap nheap))
;; 		       nargs)))))


(declaim (ftype (function (list fixnum) simple-array) construct-slices))
(defun construct-slices (args n)
  (loop with a = (make-array n)
	with b = (vec-num-bits (car args))
	for i from 0 below n
	for h = (create-heap (lambda (x y)
			       (< (formula-depth x)
				  (formula-depth y))))
	when (< i b) do (loop for arg of-type form-vec in args
			      for b = (get-bit arg i)
			      unless (eq b *zero*)
			        do (heap-insert h b))
	do (setf (aref a i) h)
	finally (return a)))

(declaim (ftype (function (heap) formula) xor-slice))
(defun xor-slice (slice)
  (sb-xor-form (loop with hc = (heap-count slice) 
		     for i from 0 below hc
		     collect (heap-remove slice))))


(declaim (ftype (function (heap) formula) or-slice))
(defun or-slice (slice)
  (sb-or-form (loop with hc = (heap-count slice)
		    for i from 0 below hc
		    collect (heap-remove slice))))

(declaim (ftype (function (simple-array) form-vec) add-slices))
(defun add-slices (slices)
  (let ((n (array-dimension slices 0))
	(coslice (create-heap (lambda (x y)
				(< (formula-depth x)
				   (formula-depth y))))))
    (if (= n 1)
	(let ((sum (add-slice (aref slices 0) coslice)))
	  (values sum (or-slice coslice)))		   
      (let ((vec (new-vec n)))
	(loop for i from 0 below (1- n)
	      do (vec-set-bit vec i (add-slice (aref slices i) (aref slices (1+ i))))
	      finally (return (progn (vec-set-bit vec (1- n) (xor-slice (aref slices (1- n))))
				     (make-unique-vec vec))))))))


(declaim (ftype (function (list) form-vec) mod+-form))
(defun mod+-form (args)
  (add-slices (construct-slices args
				(vec-num-bits (car args)))))

(declaim (ftype (function ((cons form-vec (cons form-vec null))) form-vec) mod--form))
(defun mod--form (args)
  (let ((slices (construct-slices (list (first args)
					(not-form (second args)))
				  (vec-num-bits (car args)))))
    (heap-insert (aref slices 0) *one*)
    (add-slices slices)))

(declaim (ftype (function (list) form-vec) +-form))
(defun +-form (args)
  (mod+-form (mapcar (lambda (x)
		       (ext-form (+ (vec-num-bits (car args))
				    (ceil-log (length args)))
				 x))
		     args)))

(declaim (ftype (function ((cons form-vec (cons form-vec null))) form-vec) bin---form))
(defun bin---form (args)
  (mod--form (mapcar (lambda (x)
		       (ext-form (+ (vec-num-bits (car args))
				    (ceil-log (length args)))
				 x))
		     args)))

(declaim (ftype (function ((cons form-vec null)) form-vec) unary---form))
(defun unary---form (args)
  (let* ((arg (car args))
	 (bits (1+ (vec-num-bits arg)))
	 (slices (construct-slices (list (not-form (ext-form bits arg)))
				  bits)))
    (heap-insert (aref slices 0) *one*)
    (add-slices slices)))

(declaim (ftype (function (list) form-vec) add-form))
(defun add-form (args)
  (let* ((m (vec-num-bits (first args)))
	 (sum (+-form args))
	 (n (vec-num-bits sum)))
    (cat-form (list (sb-and-form (loop for i from m below n
				       collect (sb-equiv-form (list (get-bit sum i)
								    (get-bit sum (1- m))))))
		    (get-bits sum 0 m)))))

(declaim (ftype (function ((cons form-vec (cons form-vec null))) form-vec) sub-form))
(defun sub-form (args)
  (let ((m (vec-num-bits (first args)))
	(diff (bin---form args)))
    (cat-form (list (sb-equiv-form (list (get-bit diff m)
					 (get-bit diff (1- m))))
		    (get-bits diff 0 m)))))

(declaim (ftype (function ((cons form-vec null)) form-vec) neg-form))
(defun neg-form (args)
  (let ((m (vec-num-bits (first args)))
	(neg (unary---form args)))
    (cat-form (list (sb-equiv-form (list (get-bit neg m)
					 (get-bit neg (1- m))))
		    (get-bits neg 0 m)))))

(declaim (ftype (function ((cons form-vec null)) form-vec) inc-form))
(defun inc-form (args)
  (let* ((arg (first args))
	 (bits (1+ (vec-num-bits arg)))
	 (slices (construct-slices (list (ext-form bits arg))
				  bits)))
    (heap-insert (aref slices 0) *one*)
    (add-slices slices)))

(declaim (ftype (function ((cons form-vec null)) form-vec) dec-form))
(defun dec-form (args)
  (let* ((arg (first args))
	 (bits (1+ (vec-num-bits arg)))
	 (slices (construct-slices (list (ext-form bits arg))
				  bits)))
    (dotimes (i bits) (heap-insert (aref slices i) *one*))
    (add-slices slices)))

;;*** mul/* ***

(declaim (ftype (function (fixnum list) simple-array) construct-mult-slices-aux))
(defun construct-mult-slices-aux (n args)
  (let ((na (make-array n :element-type 'list :initial-element nil)))
    (cond ((endp args) na)
	  ((endp (rest args))
	   (loop with arg of-type form-vec = (first args)
		 for i from 0 below n
		 for biti = (get-bit arg i)
		 if (eq biti *zero*) do (setf (aref na i) nil)
		 else do (setf (aref na i) `((,biti)))
		 finally (return na)))
	  (t
	   (let ((a (construct-mult-slices-aux n (cdr args)))
		 (ca (the form-vec (first args))))
	     (dotimes (i n na)
	       (let ((nai nil))
		 (dotimes (j (1+ i))
		   (let ((z (get-bit ca (- i j))))
		     (unless (eq z *zero*)
		       (dolist (x (aref a j))
			 (setf nai (acons z x nai))))))
		 (setf (aref na i) nai))))))))

;; (defun any-pair (slice)
;;   (let ((elts (loop until (heap-empty-p slice)
;; 		    collect (heap-remove slice))))
;;     (loop for x in elts
;; 	  for rst on (rest elts)
;; 	  append (loop for y in rst collect (sb-and-form (list x y))) into args
;; 	  finally (sb-or-form args))))

(declaim (ftype (function (list) simple-array) construct-mult-slices))
(defun construct-mult-slices (args)
  (let* ((n (vec-num-bits (first args)))
	 (s (construct-mult-slices-aux n args))
	 (slices (make-array n)))
    (loop for i from 0 below n
	  for h = (create-heap (lambda (x y) (< (formula-depth x)
						(formula-depth y))))
	  do (loop for a of-type list in (aref s i)
		   for af = (sb-and-form a)
		   unless (eq af *zero*)
                     do (heap-insert h af))
	  do (setf (aref slices i) h)
	  finally (return slices))))  

(declaim (ftype (function (list) form-vec) *-form))
(defun *-form (args)
  (let* ((n (loop for a in args sum (vec-num-bits a)))
	 (slices (construct-mult-slices (mapcar (lambda (x) (ext-form n x)) args))))
    (add-slices slices)))

(declaim (ftype (function (list) form-vec) mod*-form))
(defun mod*-form (args)
      (add-slices (construct-mult-slices args)))

(declaim (ftype (function (list) form-vec) mult-form))	 
(defun mult-form (args)
  (loop with bits = (vec-num-bits (car args))
	with p = (*-form args)
	with top-bit = (get-bit p (1- bits))
	for i from bits below (vec-num-bits p)
	collect (sb-equiv-form (list top-bit (vec-get-bit p i))) into ob-args
	finally (return (cat-form (list (sb-and-form ob-args) (get-bits p 0 (1- bits)))))))

;;   (multiple-value-bind
;;       (prod co)
;;       (add-slices (construct-mult-slices args))
;;     (cat-form (list co prod))))


;; (declaim (ftype (function (list) list) construct-slices))
;; (defun construct-slices (args)
;;   (declare (type list args))
;;   (loop for bit from 0 to (1- (vec-num-bits (car args)))
;; 	collect (loop for arg of-type form-vec in args
;; 		      for b = (get-bit arg bit)
;; 		      unless (eq b *zero*)
;; 		      collect b)))

;; (declaim (ftype (function ((member inc dec sub add - +) list) list) add-form1))
;; (defun add-form1 (op args)
;;   (declare (type (member inc dec sub add - +) op) (type list args))
;;   (let* ((bits (vec-num-bits (car args)))
;; 	 (dbits (+ bits (the integer (ceil-log (length args))))))
;;     (case op
;;       (inc (let ((slices (construct-slices args)))
;; 	      (add-slices (cons (cons *one* (car slices))
;; 			       (cdr slices))
;; 			 dbits)))
;;       (dec (add-slices (construct-slices (cons (ext-form bits *one*) args))
;; 		       dbits))
;;       (sub (let ((slices (if (endp (cdr args))
;; 			     (construct-slices (list (not-form
;; 						      (first args))))
;; 			   (construct-slices (list (first args)
;; 						   (not-form (second args)))))))
;; 	     (add-slices (cons (cons *one* (car slices))
;; 			      (cdr slices)) dbits)))
;;       (add (add-slices (construct-slices args) dbits))
;;       (- (cdr (add-form1 'sub (mapcar (lambda (x) (ext-form (1+ bits) x)) args))))
;;       (+ (cdr (add-form1 'add (mapcar (lambda (x) (ext-form (1+ bits) x))
;; 				      args)))))))

;; (declaim (ftype (function ((member inc dec sub add - +) list) form-vec) add-form))
;; (defun add-form (op args)
;;   (declare (type (member inc dec sub add - +) op) (type list args))
;;   (let* ((nargs (add-form1 op args)))
;;     (if (endp (cdr nargs))
;; 	(the formula (car nargs))
;;       (cat-form nargs))))
	
;***foldr/foldl***

(declaim (ftype (function ((cons symbol (cons form-vec null))) formula) foldl-form))
(defun foldl-form (args)
  (declare (type list args))
  (let* ((op (first args))
	 (arg (second args))
	 (bits (vec-num-bits arg))
	 (dtp (1- bits))
	 (nform nil))
    (dotimes (i bits nform)
      (let ((b (get-bit arg (- dtp i))))
	;(format t "~&b: ~a" b)
	(if (= i 0)
	    (setf nform b)
	  (setf nform
		(make-formula :fn op
			      :args (list b nform))))))))

(declaim (ftype (function ((cons symbol (cons form-vec null))) formula) foldr-form))
(defun foldr-form (args)
  (let ((op (first args))
	(arg (second args))
	;(dtp (1- type))
	(nform nil))
    (dotimes (i (vec-num-bits arg) nform)
      (let ((b (get-bit arg i)))
	(if (= i 0)
	    (setf nform b)
	  (setf nform
		(make-formula :fn op
			      :args (list nform b))))))))

;***if***

(declaim (ftype (function (formula formula formula) formula) sb-if-form))
(defun sb-if-form (ifexp thenexp elseexp)
  (cond ((eq ifexp thenexp) (sb-or-form (list ifexp elseexp)))
 	((negation-of ifexp thenexp) (sb-and-form (list (sb-not-form ifexp) elseexp)))
 	((eq ifexp elseexp) (sb-and-form (list ifexp thenexp)))
 	((negation-of ifexp elseexp) (sb-or-form (list (sb-not-form ifexp) thenexp)))
	((eq thenexp elseexp) thenexp)
	((eq thenexp *one*) (sb-or-form (list ifexp elseexp)))
	((eq thenexp *zero*) (sb-and-form (list (sb-not-form ifexp) elseexp)))
	((eq elseexp *one*) (sb-or-form (list (sb-not-form ifexp) thenexp)))
	((eq elseexp *zero*) (sb-and-form (list ifexp thenexp)))
	((negation-of thenexp elseexp) (sb-equiv-form (list ifexp thenexp)))
	((eq (formula-fn thenexp) 'not)
	 (sb-not-form (make-unique-formula :fn 'if
					   :args (list ifexp
						       (first (formula-args thenexp))
						       (sb-not-form elseexp)))))
	(t (make-unique-formula :fn 'if :args (list ifexp thenexp elseexp)))))

(declaim (ftype (function (formula formula formula) formula) mv-if-form))
(defun mv-if-form (ifexp thenexp elseexp)
  (mv-form (formula-type thenexp)
	   (loop for texp in (formula-args thenexp)
		 and eexp in (formula-args elseexp)
		 collecting (if-form (list ifexp texp eexp)))))
  

(declaim (ftype (or (function ((cons formula (cons formula (cons formula null)))) formula)
		    (function ((cons formula (cons vec (cons vec null)))) vec))
 		if-form)
	 (inline if-form))
(defun if-form (args)
  (let* ((switch? (eq (formula-fn (first args)) 'not))
	 (ifexp (if switch? (sb-not-form (first args)) (first args)))
	 (thenexp (if switch? (third args) (second args)))
	 (elseexp (if switch? (second args) (third args))))
    (cond ((eq ifexp *one*) thenexp)
	  ((eq ifexp *zero*) elseexp)
	  ((eq thenexp elseexp) thenexp)
	  ((vec-p thenexp)
	   (let* ((vb (vec-num-bits thenexp))
		  (vec (new-vec vb)))
	     (dotimes (i vb (make-unique-vec vec))
	       (vec-set-bit vec i (sb-if-form ifexp (get-bit thenexp i) (get-bit elseexp i))))))
	  ((eq (car (formula-type thenexp)) 'mem)
	   (make-unique-formula :fn 'if :type (formula-type thenexp) :args args))
	  ((eq (formula-fn thenexp) 'mv)
	   (mv-if-form ifexp thenexp elseexp))
	  (t
	   (sb-if-form ifexp thenexp elseexp)))))

;***cond***

(declaim (ftype (function (form-vec list) form-vec) cond-form1))
(defun cond-form1 (default args)
  (if (endp args)
      default
    (let* ((first-pair (car args))
	   (ff (the formula (first first-pair)))
	   (sf (the form-vec (second first-pair))))
      (cond ((eq ff *one*) sf)
	    ((eq ff *zero*) (cond-form1 default (cdr args)))
	    (t
	     (the form-vec (if-form (list ff 
					  sf 
					  (cond-form1 default (cdr args))))))))))

(declaim (ftype (function ((cons (cons formula (cons form-vec null)) list)) form-vec) cond-form))
(defun cond-form (args)
  (let ((sca (second (car args))))
    (cond ((vec-p sca)
	   (cond-form1 (cat-form (n-copies *zero* (vec-num-bits sca)))
		       args))
	  ((eq (formula-fn sca) 'mv)
	   (cond-form1 *junk* args))
	  (t
	   (cond-form1 *zero* args)))))

;***<***

(declaim (ftype (function (form-vec form-vec fixnum formula) formula) <-form-loop))
(defun <-form-loop (arg1 arg2 i nform)
  (if (< i 0)
      *zero*
    (let* ((arg1i (get-bit arg1 i))
	   (tform (sb-xor-form (list arg1i (get-bit arg2 i)))))
      (cond ((eq tform *zero*) (<-form-loop arg1 arg2 (1- i) nform))
	    ((eq tform *one*) (sb-equiv-form (list nform arg1i)))
	    (t (if-form (list tform
			      (sb-equiv-form (list nform arg1i))
			      (<-form-loop arg1 arg2 (1- i) nform))))))))

(declaim (ftype (function ((cons form-vec (cons form-vec null))) formula) <-form))
(defun <-form (args)
  (let* ((arg1 (first args))
	 (arg2 (second args))
	 (bits (vec-num-bits arg1))
	 (nform (sb-or-form (list (get-bit arg1 (1- bits)) (get-bit arg2 (1- bits))))))
    (<-form-loop arg1 arg2 (1- bits) nform)))

;***shift***

(declaim (ftype (function (form-vec fixnum) form-vec) shift-by-constant))
(defun shift-by-constant (form const)
  (cond ((= const 0) form)
	((vec-p form)
	 (let* ((bits (vec-num-bits form))
		(vec (new-vec bits)))
	   (dotimes (i bits (make-unique-vec vec))
	     (if (and (>= i const)
		      (< i (+ bits const)))
		 (vec-set-bit vec i (get-bit form (- i const)))
	       (vec-set-bit vec i *zero*)))))
	(t
	 *zero*)))

(declaim (ftype (function ((cons form-vec (cons fixnum null))) form-vec) shift-left-form))
(defun shift-left-form (args)
  (shift-by-constant (first args) (second args)))

(declaim (ftype (function ((cons form-vec (cons fixnum null))) form-vec) shift-right-form))
(defun shift-right-form (args)
  (shift-by-constant (first args) (- (second args))))

(declaim (ftype (function (form-vec fixnum) form-vec) cshift-by-constant))
(defun cshift-by-constant (form const)
  (cond ((= const 0) form)
	((vec-p form)
	 (let* ((bits (vec-num-bits form))
		(vec (new-vec bits)))
	   (dotimes (i bits (make-unique-vec vec))
	     (vec-set-bit vec i (get-bit form (mod (- i const) bits))))))
	(t form)))

(declaim (ftype (function ((cons form-vec (cons fixnum null))) form-vec) cshift-left-form))
(defun cshift-left-form (args)
; PETE: commented the declare below out due to errors it was
; giving. try it with (compute (<<< 0b0001 2)) in sbcl
;  (declare (type (cons formula (cons integer null)) args))
  (cshift-by-constant (first args) (second args)))

(declaim (ftype (function ((cons form-vec (cons fixnum null))) form-vec) cshift-right-form))
(defun cshift-right-form (args)
; PETE: commented the declare below out due to errors it was
; giving. try it with (compute (>>> 0b0001 2)) in sbcl
;  (declare (type (cons formula (cons integer null)) args))
  (cshift-by-constant (first args) (- (second args))))

;***user defined functions***

(declaim (ftype (function (symbol (or fixnum symbol)) symbol) funct-var-name))
(defun funct-var-name (name bit)
  (read-from-string (format nil "_~A_~A" name bit)))


;***local***

(declaim (ftype (function (fixnum) form-vec) junkform))
(defun junk-form (bits)
  (declare (type fixnum bits))
  (if (= bits 1)
      *junk*
    (let ((vec (new-vec bits)))
      (dotimes (i bits (make-unique-vec vec)) (vec-set-bit vec i *junk*)))))

;sets bits bit0 to bit1 in var to be value, updating the environment, env, appropriately.
(declaim (ftype (function (symbol list fixnum fixnum form-vec) form-vec)))
(defun setbits (var env bit0 bit1 value)
  (let* ((pr (assoc var env))
	 (old-val (the form-vec (cdr pr))))
    (if (vec-p old-val)
	(dotimes (i (- (1+ bit1) bit0) old-val)
	  (vec-set-bit old-val (+ i bit0) (get-bit value i)))
      (setf (cdr pr) value))))

(declaim (ftype (function (symbol list fixnum form-vec) form-vec) setbit))
(defun setbit (var env bit value)
  (declare (type symbol var) (type list env)
	   (type fixnum bit) (type form-vec value))
  (setbits var env bit bit value))

(declaim (ftype (function (symbol form-type list) 
			  (cons (cons symbol form-vec) list))
		add-var-to-env))
(defun add-var-to-env (var type env)
  (declare (type symbol var) (type form-type type) (type list env))
   (acons var (junk-form (type-bits type)) env))

(declaim (ftype (function (list list list) list) local-form-mv))	
(defun local-form-mv (bargs mv env)
  (if (endp bargs)
      env
    (let* ((mvform (the formula (car bargs)))
	   (mvargs (formula-args mvform))
	   (val (the form-vec (car mv))))
      (case (formula-fn mvform)
	(new-mv-binding (local-form-mv (cdr bargs) (cdr mv) (acons (first mvargs) val env)))
	(var-mv-binding (setbits (first mvargs) env 0 (1- (formula-bits mvform)) val)
			(local-form-mv (cdr bargs) (cdr mv) env))
	(bit-mv-binding (setbit (first mvargs) env (second mvargs) val)
			(local-form-mv (cdr bargs) (cdr mv) env))
	(bits-mv-binding (setbits (first mvargs) env (second mvargs) (third mvargs) val)
			 (local-form-mv (cdr bargs) (cdr mv) env))))))

;; the next few functions are mutually recursive with simplify1, so we
;; put the type declaration up here.
(declaim (ftype (function (formula list list list) form-vec) simplify1))

(declaim (ftype (function ((or (cons list (cons list (cons formula null)))
			       (cons list (cons formula null)))
			   list list list)
			  form-vec)
		local-form))
(defun local-form (args env nenv functs)
  (let ((env env))
    (dolist (v (car args) nil)
      (if (consp v)
	  (setf env (add-var-to-env (first v) (rest v) env))
	(setf env (add-var-to-env v '(bv 1) env))))
    (let ((args (cdr args)))
      (dolist (bform (car args) (progn #|(format t "~&env: ~A~%" env)|#
				  (the form-vec (simplify1 (second args) env nenv functs))))
	(let* ((bargs (formula-args bform))
	       (expression (the form-vec (simplify1 (second bargs) env nenv functs))))
	  (case (formula-fn bform)
	    (new-binding 
	     (setf env (cons (cons (first bargs) expression) env))
	     #|(format t "~&bform: ~A~%set ~A to ~A~%" bform (first bargs) expression)|#)
	    (var-binding (setbits (first bargs) env 0 (1- (formula-bits bform)) expression))
	    (bit-binding 
	     (let ((a (car bargs)))
	       (setf env (setbit (first a) env (second a) expression))))
	    (bits-binding 
	     (let ((a (car bargs)))
	       (setf env (setbits (first a) env (second a) (third a) expression))))
	    (cat-binding
	     (let ((bit1 0)
		   (bit0 0))
	       (dolist (mvform (reverse (car bargs)) nil)
		 (setf bit0 bit1)
		 (setf bit1 (+ bit1 (formula-bits mvform)))
		 (let ((exp-bits (if (= (- bit1 bit0) 1)
				     (get-bit expression bit0)
				   (get-bits expression bit0 (1- bit1))))
		       (mvargs (formula-args mvform)))
		   (case (formula-fn mvform)
		     (new-cat-binding (setf env (cons (cons (elt mvargs 0) exp-bits) env)))
		     (var-cat-binding (setbits (first mvargs) env 0 (1- (vec-num-bits mvform)) exp-bits))
		     (bit-cat-binding (setbit (elt mvargs 0) env (elt mvargs 1) exp-bits))
		     (bits-cat-binding (setbits (first mvargs) 
						env 
						(second mvargs) 
						(third mvargs) 
						exp-bits)))))))
	    (mv-binding (setf env (local-form-mv (first bargs)
						 (formula-args expression)
						 env)))))))))

;--------------------------------------------------------------------

(declaim (ftype (function (list list list list) list) simplify-list))
(defun simplify-list (list env nenv functs)
  (mapcar #'(lambda (x) 
	      (if (formula-p x)
		  (simplify1 x env nenv functs)
		x))
	  list))

(declaim (ftype (function (symbol form-type list list list list) form-vec) simplify1-step))
(defun simplify1-step (fn tp args env nenv functs)
  (case fn
    (ext       (ext-form (type-bits tp) (first args)))
    (not       (not-form (car args)))
    (bit       (apply #'get-bit args))
    (bits      (get-bits (first args) (second args) (third args)))
    (get       (apply #'get-form args))
    (set       (apply #'set-form args))
    (=         (equals-form args))
    (<         (<-form args))
    (<->       (equiv-form args))
    (and       (and-form args))
    (or        (or-form args))
    (->        (or-form (list (not-form (first args)) (second args))))
    (cat       (cat-form args))
    (xor       (xor-form args))
    (inc       (inc-form args))
    (dec       (dec-form args))
    (+         (+-form args))
    (-         (if (endp (cdr args)) (unary---form args) (bin---form args)))
    (add       (add-form args))
    (sub       (sub-form args))
    (neg       (neg-form args))
    (mod+      (mod+-form args))
    (mod-      (mod--form args))
    (*         (*-form args))
    (mult      (mult-form args))
    (mod*      (mod*-form args))
    (if        (the form-vec (if-form args)))
    (cond      (cond-form args))
    (<<        (shift-left-form args))
    (>>        (shift-right-form args))
    (<<<       (cshift-left-form args))
    (>>>       (cshift-right-form args))
    (foldr     (let ((nform (foldr-form args)))
		 (simplify1-step (formula-fn nform)
				 (formula-type nform)
				 (formula-args nform) 
				 env nenv functs)))
    (foldl     (let ((nform (foldl-form args)))
		 (simplify1-step (formula-fn nform)
				 (formula-type nform)
				 (formula-args nform) 
				 env nenv functs)))
    (mv        (mv-form tp args))
    (_zero_mem (make-unique-formula :fn fn
				    :type tp
				    :args args))
    (otherwise (let ((funct (cdr (assoc fn functs)))
		     (fenv nil))
		 (loop for arg in args
		       for param in (funct-params funct)
		       do (setf fenv (acons (car param) arg fenv)))
		 (the form-vec (simplify1 (funct-body funct) fenv nil functs))))))


;this is the meat of the simplification algorithm.
;form is the formula to be simplified
;env is the variable environment.
;functs contains the function definitions
(defun simplify1 (form env nenv functs)
  (let ((fn (formula-fn form))
	(tp (formula-type form))
	(args (formula-args form)))
    (cond ((eq fn 'const) (const-form args))
	  ((or (eq fn 'var) (eq fn 'def)) (var-form form env))
	  ((eq fn 'next) (simplify1 (car (formula-args form)) nenv nil functs))
	  ((eq fn 'local) (local-form args env nenv functs))
	  ((eq fn 'cond) (let ((nargs (mapcar (lambda (x)
						(simplify-list x env nenv functs))
					      args)))
			   (cond-form nargs)))
	  (t
	   (simplify1-step fn
			   tp
			   (simplify-list args env nenv functs)
			   env
			   nenv
			   functs)))))


(declaim (ftype (function (formula list list list) form-vec) simplify1))
(defun simplify (form env nenv functs)
  (simplify1 form env nenv functs))

;; given an environment, env, a list of variable definitions, and a
;; "step" (usually 'this or 'next), adds variables to the environment.
(declaim (ftype (function (list t list) list) vars-to-env-tail))
(defun vars-to-env-tail (vars step env)
  (let ((env env))
    (dolist (vardef vars env)
      (let* ((name (the symbol (first vardef)))
	     (tp (the form-type (rest vardef))))
	(cond ((equal tp `(bv 1))
	       (setf env 
		     (acons name 
			    (make-unique-formula :fn 'var :args (list name step 0))
			    env)))
	      ((eq (car tp) 'mem)
	       (setf env (acons name
				(make-unique-formula :fn 'var :type tp :args (list name step))
				env)))
	      (t
	       (let ((vec (new-vec (type-bits tp))))
		 (dotimes (i (type-bits tp) (setf env (acons name 
							     (make-unique-vec vec)
							     env)))
		   (vec-set-bit vec i (make-unique-formula :fn 'var :args (list name step i)))))))))))

(declaim (ftype (function (list t) list) vars-to-env))
(defun vars-to-env (vars step)
  (vars-to-env-tail vars step nil))

;a top-level simplification function. use this when you don't have
;anything but a variable definitions list to put in your environment.
(declaim (ftype (function (formula list list list) form-vec) tl-simplify))
(defun tl-simplify (form vars defs functs)
  (let ((env (vars-to-env-tail vars 'this defs))
	(nenv (vars-to-env vars 'next)))
      (simplify form env nenv functs)))

(declaim (ftype (function (formula list list) form-vec) simplify-funct))
(defun simplify-funct (body params functs)
  (let ((env nil))
    (dolist (var params (simplify body env nil functs))
      (let* ((name (first var))
	     (tp (cdr var)))
	(cond ((equal tp '(bv 1))
	       (setf env (acons name 
				(make-unique-formula :fn 'var
						     :args (list (funct-var-name name 0)))
				env)))
	      ((eq (car tp) 'mem)
	       (setf env (acons name
			      (make-unique-formula :fn 'var
						   :type tp
						   :args (list (funct-var-name name 'mem)))
			      env)))
	      (t
	       (let ((vec (new-vec (type-bits tp))))
		 (dotimes (i (type-bits tp) (setf env (acons name
							     (make-unique-vec vec)
							     env)))
		   (vec-set-bit vec i (make-unique-formula :fn 'var
							   :args (list (funct-var-name name i))))))))))))

;simplifies the definitions section of the desc.
(declaim (ftype (function (desc) list) simplify-defs))
(defun simplify-defs (desc)
  (when (desc-defs desc)
    (let ((env (vars-to-env (desc-vars desc) 0))
	  (ndefs nil))
      (dolist (def (desc-defs desc) (setf (desc-defs desc) ndefs))
	(let ((ndef (cons (car def) (simplify (cdr def) env nil (desc-functs desc)))))
	  (setf ndefs (cons ndef ndefs))
	  (setf env (cons ndef env)))))))

(declaim (ftype (function (list) fixnum) add-list))
(defun add-list (lst)
  (loop for x of-type fixnum in lst
	sum x))

(declaim (ftype (function (t) fixnum) count-forms-1))
(defun count-forms-1 (form)
  (cond ((or (not (formula-p form))
	     (formula-slot2 form))
	 0)
	((eq (formula-fn form) 'var) 
	 (setf (formula-slot2 form) t)
	 1)
	(t 
	 (setf (formula-slot2 form) t)
	 (1+ (add-list (mapcar #'count-forms-1 (formula-args form)))))))

(declaim (ftype (function (formula) fixnum) count-forms))
(defun count-forms (form)
  (let ((num (count-forms-1 form)))
    (clear-slot2 form)
    num))

(declaim (ftype (function (desc) fixnum) desc-count-forms))
(defun desc-count-forms (desc)
  (let ((sum 0))
    (dolist (f (desc-functs desc))
      (declare (type fixnum sum))
      (setf sum (+ sum (count-forms (funct-body (cdr f))))))
    (dolist (d (desc-defs desc) (+ sum
				   (count-forms (desc-init desc))
				   (count-forms (desc-trans desc))
				   (count-forms (desc-spec desc))))
      (declare (type fixnum sum))
      (setf sum (+ sum (count-forms (cdr d)))))))

(declaim (ftype (function (list) null) simplify-functions))
(defun simplify-functions (functs)
  (dolist (f (mapcar #'cdr functs))
      (format t "~&simplifying function ~a~%" (funct-name f))
      (setf (funct-body f)
	    (simplify-funct (funct-body f) (funct-params f) functs))
      (setf *ftrie* nil)
      (clrhash *fhash*)))

;simplifies the description desc, negating the spec, 
;since we are ultimately searching for a counterexample
(declaim (ftype (function (desc) desc) simplify-desc))
(defun simplify-desc (desc)
  (setf *ftrie* nil)
  (clrhash *fhash*)
  (setf *fc* -4)
  (let ((functs (desc-functs desc)))
    (simplify-defs desc)
    (let ((vars (desc-vars desc))
          (defs (desc-defs desc)))
      (format t "~&simplifying init~%")
      (setf (desc-init desc) 
            (tl-simplify (desc-init desc) vars defs functs))
      (format t "~&simplifying trans~%")
      (setf (desc-trans desc) 
            (tl-simplify (desc-trans desc) vars defs functs))
      (format t "~&simplifying spec~%")
      (setf (desc-spec  desc) 
            (if (member (formula-fn (desc-spec desc)) '(AG AF))
                (make-formula :fn (if (eq (formula-fn (desc-spec desc)) 'AG) 'EF 'EG)
                              :args (list (sb-not-form (tl-simplify (car (formula-args (desc-spec desc)))
								    vars
								    defs
								    functs))))
              (sb-not-form (tl-simplify (desc-spec desc) vars defs functs))))))
  (clrhash *fhash*)
  (setf *ftrie* nil)
  desc)

(declaim (ftype (function (formula desc) form-vec) simplify-form))
(defun simplify-form (form desc)
  (setf *ftrie* nil)
  (clrhash *fhash*)
  (setf *fc* -4)
  (let ((functs (desc-functs desc)))
    (let ((vars (desc-vars desc)))
      (format t "~&simplifying formula~%")
      (let ((nform (simplify form (vars-to-env vars 0) nil functs)))
	(clrhash *fhash*)
	(setf *ftrie* nil)
	nform))))